
#if defined (__dsPIC33F__)
    #include <p33Fxxxx.h>
#elif defined (__PIC24H__)
    #include <p24Hxxxx.h>
#elif defined (__PIC24F__)
    #include <p24Fxxxx.h>
#else
    #error Selected processor not supported
#endif

#include "Simple EEPROM Emulation.h"

// User constant validation
#if NUM_DATA_EE_PAGES < 2
    #error Minimum number of program memory pages is 2
#endif


//Data EE info stored in PM in following format
//  Status in first two locations of PM page,
//  8-bit DEE Address (odd address, low byte) 16-bit DEE data (even address)
unsigned char emulationPages[NUM_DATA_EE_PAGES][NUMBER_OF_INSTRUCTIONS_IN_PAGE * 2]
    __attribute__ ((space(psv), aligned(NUMBER_OF_INSTRUCTIONS_IN_PAGE * 2), noload));

#define DEE_PAGE_SIZE (sizeof(emulationPages[0]))

#if __C30_VERSION__ > 301
    #define DEE_PAGE_TBL(page) ((__builtin_tbladdress(&emulationPages) + (DEE_PAGE_SIZE * (page))) >> 16)
    #define DEE_PAGE_OFFSET(page) ((__builtin_tbladdress(&emulationPages) + (DEE_PAGE_SIZE * (page))) & 0xFFFF)
#else
    #warning "Please upgrade your C30 compiler"
    #define DEE_PAGE_TBL(page) ((((((unsigned long)__builtin_tblpage(&emulationPages)) << 16) + __builtin_tbloffset(&emulationPages)) + \
                                 (DEE_PAGE_SIZE * (page))) >> 16)
    #define DEE_PAGE_OFFSET(page) ((((((unsigned long)__builtin_tblpage(&emulationPages)) << 16) + __builtin_tbloffset(&emulationPages)) + \
                                    (DEE_PAGE_SIZE * (page))) & 0xFFFF)
#endif

/************************************************************************
UnlockWrite

This routine saves the current CPU priority and sets it the highest
user level of 7. It calls an assembly routine to perform an unlock
sequence and sets the WR bit in NVMCON. The WR bit is polled until it
clears indicating the flash operation is complete. The previous CPU
priority is restored.

Parameters:		None
Return:			None
Side Effects:	None
************************************************************************/
void UnlockWrite(void) {
    unsigned int sh_SR;

    SET_AND_SAVE_CPU_IPL(sh_SR, 7);

    UnlockPM();

    RESTORE_CPU_IPL(sh_SR);

    return;
}

/************************************************************************
ErasePage

This routine erases the selected page.

Parameters:		Page number
Return:			None
Side Effects:	Loads NVCOM with erase opcode
************************************************************************/
void ErasePage(unsigned char page) {
    unsigned int pmOffset;           //Current array (page) offset of selected element (PM 16-bit word)
    int savedTBLPAG;        //Context save of TBLPAG value. Current and packed page are on same page.

    savedTBLPAG = TBLPAG;

    // Point to proper TBLPAG and offset
    TBLPAG = DEE_PAGE_TBL(page);

    NVMCON = ERASE;

    pmOffset = DEE_PAGE_OFFSET(page);

    WritePMLow(pmOffset, pmOffset);

    UnlockWrite();

    TBLPAG = savedTBLPAG;

    return;
}

static unsigned short Read12BitWord(unsigned char Page, unsigned short Index) {
  int savedTBLPAG;        //Context save of TBLPAG value. Current and packed page are on same page.
  unsigned short Offset, Ret;

  savedTBLPAG = TBLPAG;

  TBLPAG = DEE_PAGE_TBL(Page);
  Offset = DEE_PAGE_OFFSET(Page) + (Index&~1);
  if( Index&1 ) {
    Ret = ((ReadPMLow(Offset)>>12)|(((unsigned short)ReadPMHigh(Offset))<<4))&0xFFF;
  } else {
    Ret = ReadPMLow(Offset)&0xFFF;
  }

  TBLPAG = savedTBLPAG;
  return Ret;
}

static void Write12BitWord(unsigned char Page, unsigned short Index, unsigned short Value) {
  int savedTBLPAG;        //Context save of TBLPAG value. Current and packed page are on same page.
  unsigned int Offset;

  savedTBLPAG = TBLPAG;
  NVMCON = PROGRAM_WORD;

  TBLPAG = DEE_PAGE_TBL(Page);
  Offset = DEE_PAGE_OFFSET(Page) + (Index&~1);
  if( Index&1 ) {
    unsigned short temp = ReadPMLow(Offset);
    WritePMLow((temp&0x0FFF)|(Value<<12), Offset);
    WritePMHigh(Value>>4, Offset);
  } else {
    WritePMLow(Value|0xF000, Offset);
    WritePMHigh(0xFF, Offset);
  }

  UnlockWrite();

  TBLPAG = savedTBLPAG;
}

unsigned short ActivePage, ActiveIndex, LastBlankPage;
unsigned char NoDataInFlash;

/************************************************************************
DataEEInit

Parameters:		None
Return:			-
Side Effects:	Memory pages may be erased
************************************************************************/
void DataEEInit(void) {
  unsigned short i, Value, LatestOccupiedPage = 0xFFFF;

  // Find the latest page in use. If there isn't one, pick page 0.
  for( i = 0; i < NUM_DATA_EE_PAGES; ++i ) {
    Value = Read12BitWord(i, 0);
    if( Value != 0xFFF ) {
      if( LatestOccupiedPage == 0xFFFF )
        LatestOccupiedPage = i;
      Value = Read12BitWord(i, NUMBER_OF_INSTRUCTIONS_IN_PAGE*2-1);
      if( Value == 0xFFF )
        break;
    } else if( LatestOccupiedPage != 0xFFFF ) {
      --i;
      break;
    }
  }
  if( LatestOccupiedPage == 0xFFFF ) {
    ActivePage = 0;
    ActiveIndex = 0;
    NoDataInFlash = 1;
  } else if( i == NUM_DATA_EE_PAGES ) {
    ActivePage = NUM_DATA_EE_PAGES-1;
    ActiveIndex = NUMBER_OF_INSTRUCTIONS_IN_PAGE*2-1;
  } else {
    ActivePage = i;
    for( i = 0; i < NUMBER_OF_INSTRUCTIONS_IN_PAGE*2-1; ++i ) {
      if( Read12BitWord(ActivePage, i) == 0xFFF ) {
        --i;
        break;
      }
    }
    ActiveIndex = i;
  }

  // erase any populated pages other than the latest one
  for( i = 0; i < NUM_DATA_EE_PAGES; ++i ) {
    if( i == ActivePage )
      continue;

    Value = Read12BitWord(i, 0);
    if( Value != 0xFFF ) {
      ErasePage(i);
    }
  }

  LastBlankPage = ActivePage + NUM_DATA_EE_PAGES - 1;
  if( LastBlankPage >= NUM_DATA_EE_PAGES )
    LastBlankPage -= NUM_DATA_EE_PAGES;
}

/************************************************************************
DataEERead

Parameters:		
Return:			Data EE data or 0xFFF
Side Effects:	None
************************************************************************/
unsigned int DataEERead(unsigned int addr) {
  return Read12BitWord(ActivePage, ActiveIndex);
}

/************************************************************************
DataEEWrite

Parameters:		Data EE data
Return:			-
Side Effects:	CPU stall occurs for flash programming.
************************************************************************/
void DataEEWrite(unsigned int data) {
  if( !NoDataInFlash )
    ++ActiveIndex;
  else
    NoDataInFlash = 0;
  if( ActivePage == LastBlankPage && ActiveIndex == NUMBER_OF_INSTRUCTIONS_IN_PAGE*2-1 ) {
    ++LastBlankPage;
    if( LastBlankPage >= NUM_DATA_EE_PAGES )
      LastBlankPage -= NUM_DATA_EE_PAGES;
    ErasePage(LastBlankPage);
  }
  if( ActiveIndex == NUMBER_OF_INSTRUCTIONS_IN_PAGE*2 ) {
    ++ActivePage;
    if( ActivePage >= NUM_DATA_EE_PAGES )
      ActivePage -= NUM_DATA_EE_PAGES;
    ActiveIndex = 0;
  }

  Write12BitWord(ActivePage, ActiveIndex, data);
}
